/**
 * \file: sdc_intern.h
 *
 * \brief : Definiton of SDC internal function and types common for all
 * architectures.
 * <b>SDC private header</b>
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \defgroup internal Internal functions
 * Internal functions ...
 */

#ifndef __SDC_LIB_INTERN_H_
#define __SDC_LIB_INTERN_H_

#include <sdc.h>
#include <sdc_session.h>
#include <sdc_op_common.h>
#include <sdc_keystore_keys.h>

#define ARRAY_SIZE(a) (sizeof(a) / sizeof(a[0]))

/**
 * Static assert as used by
 * https://developer.gnome.org/glib/stable/glib-Miscellaneous-Macros.html#G-STATIC-ASSERT-EXPR:CAPS
 */
#define STATIC_ASSERT_EXPR(expr) ((void) sizeof (char[(expr) ? 1 : -1]))

/* Definitions types and defaults */

/**
 * \ingroup internal
 * \brief Used to specify that this function is part of the internal API of libSDC
 *
 * These function won't be visible to applications including libSDC library */
#define SDC_INTERN_API __attribute__((__visibility__("internal")))

/**
 * \ingroup internal
 * \defgroup internal_wrap_unwrap_param init/update/finalize wrap unwrap parameters
 * structure to hold specific parameters of init/update/finalize of wrap/unwrap
 */

/**
 * \ingroup internal
 * \defgroup internal_op_sequence operation sequence
 * structure to handle crypto operation and functions sequence
 */

/**
 * \ingroup internal_op_sequence
 * \brief Enum used to keep track of current crypto operation
 */
enum sdc_op_crypto {
    SDC_OP_NO_CRYPTO = 0,  /**< No operation is currently ongoing */
    SDC_OP_ENCRYPT,        /**< encryption is in progress */
    SDC_OP_DECRYPT,        /**< decryption is in progress */
    SDC_OP_WRAP,           /**< wrap is in progress */
    SDC_OP_UNWRAP,         /**< unwrap is in progress */
    SDC_OP_SIGN,           /**< sign is in progress */
    SDC_OP_VERIFY,         /**< verify is in progress */
    SDC_OP_DGST            /**< verify is in progress */
};

/**
 * \ingroup internal_op_sequence
 * \brief Type of enum used to to keep track of current crypto operation.
 */
typedef enum sdc_op_crypto sdc_op_crypto_t;

/**
 * \ingroup internal_op_sequence
 * \brief Enum used to keep track of intermediate operation function state.
 */
enum sdc_op_state{
    SDC_NO_OP = 0,     /**< init or one shot allowed - no operation is currently ongoing and always combined with SDC_OP_NO_CRYPTO */
    SDC_OP_INITILIZED, /**< update or aad update(only in wrap/unwrap) or finalize allowed - operation is in progress and init or one shot calls will abort the current operation */
    SDC_OP_ERROR       /**< init or one shot allowed - operation ended with error and abort required in init or one shot calls */
};

/**
 * \ingroup internal_op_sequence
 * \brief Type of enum used to keep track of intermediate operation function state.
 */
typedef enum sdc_op_state sdc_op_state_t;

/**
 * \ingroup internal_op_sequence
 * \brief This structure contains type of crypto operation and system sequence status.
 *
 * This struct will be used to hold operations sequence status in order to avoid
 * conflict between the crypto operation and its multi stage functions.
 */
struct sdc_ops_sequence {
    sdc_op_crypto_t crypto_op; /**< crypto operation type */
    sdc_op_state_t op_state;  /**< operation state */
};

/**
 * \ingroup internal_op_sequence
 * \brief This structure contains crypto operation and function sequence.
 */
typedef struct sdc_ops_sequence sdc_ops_sequence_t;

/**
 * \ingroup internal
 * \defgroup internal_perm Permissions
 * Types, defines to model permissions
 */

/**
 * \ingroup internal_perm
 * \brief internally used structure to model permissions
 *
 * Currently only the group is stored.
 * Later it will contain a list of flags to specify which operations are allowed
 * with the key this permission structure is assigned to.
 */
struct sdc_permissions {
    uid_t uid; /**< ID of the user allowed to perform operations with the corresponding key */
    gid_t gid; /**< ID of the group allowed to perform operations with the corresponding key */
    sdc_perm_bmsk_t perms_owner; /**< permissions of the user using this key */
    sdc_perm_bmsk_t perms_group; /**< permissions of the group using this key */
    sdc_perm_bmsk_t perms_others; /**< permissions of someone being neither a member of group or user */
    sdc_perm_bmsk_t perms_inherit_owner; /**< when importing a key using this key, this mask defines the upper limit of \ref perms_owner and \ref perms_inherit_owner of the imported key */
    sdc_perm_bmsk_t perms_inherit_group; /**< when importing a key using this key, this mask defines the upper limit of \ref perms_group and \ref perms_inherit_group of the imported key */
    sdc_perm_bmsk_t perms_inherit_others; /**< when importing a key using this key, this mask defines the upper limit of \ref perms_others and \ref perms_inherit_others of the imported key */
};

/**
 * \ingroup internal
 * \defgroup internal_key Key handling
 * Types/structs to handle key installation etc.
 */

/**
 * \ingroup internal_key
 * \brief Struct containing settings shared for installation etc. of builtin and keystore keys
 *
 * This struct will be used inside of architecture specific functions
 * to avoid passing a high number of arguments without changes through
 * multiple levels of hierarchy
 */
struct sdc_key_common {
    sdc_key_fmt_t fmt;  /**< format of key */
    sdc_key_len_t len;  /**< length of key */
    const sdc_permissions_t *perms; /**< perms of key */
};
/**
 * \ingroup internal_key
 * \brief Type for settings shared for installation etc. of builtin and keystore keys
 */
typedef struct sdc_key_common sdc_key_common_t;

/**
 * \ingroup internal_key
 * \brief Struct containing settings shared for installation etc. of keystore keys
 *
 * This struct will be used inside of architecture specific functions
 * to avoid passing a high number of arguments without changes through
 * multiple levels of hierarchy
 */
struct sdc_keystore_key_cfg {
    sdc_key_id_t *kid;                      /**< in/out key id */
    sdc_key_id_options_t kid_opt;           /**< select fixed or automatically determined kid */
    sdc_key_storage_options_t key_stor_opt; /**< select persistent or volatile key */
    sdc_key_common_t common;                /**< struct containing format/length/permissions  */
};
/**
 * \ingroup internal_key
 * \brief Type for settings shared for installation etc. of keystore keys
 */
typedef struct sdc_keystore_key_cfg sdc_keystore_key_cfg_t;

/**
 * \ingroup internal_key
 * \brief Struct containing keysecret used for installation etc. of keystore keys
 *
 * This struct will be used inside of architecture specific functions
 * to avoid passing a high number of arguments without changes through
 * multiple levels of hierarchy
 */
struct sdc_keysecret {
    sdc_keysecret_enc_t enc; /**< key encoding */
    const uint8_t *secret;   /**< key secret blob, NULL for creating random key */
    const size_t secret_len; /**< length of key secret blob */
};
/**
 * \ingroup internal_key
 * \brief Type for keysecret used for installation etc. of keystore keys
 */
typedef struct sdc_keysecret sdc_keysecret_t;

/**
 * \ingroup internal_key
 * \brief Struct to collect supported key length and format of a operation
 */
struct sdc_key_desc {
    sdc_key_len_t dflt_key_len;          /**< default key length of the algorithm */
    sdc_key_len_bmsk_t sup_key_lens;    /**< Bitmask to list the key length supported by the algorithm */
    sdc_key_fmt_t sup_key_fmt_protect;  /**< key format supported by the algorithm - protect operation (sign,encrypt,wrap)*/
    sdc_key_fmt_t sup_key_fmt_unprotect; /**< key format supported by the algorithm - unprotect operation (verify,decrypt,unwrap)*/
};
/**
 * \ingroup internal_key
 * \brief Type to struct \ref sdc_key_desc
 */
typedef struct sdc_key_desc sdc_key_desc_t;

/**
 * \ingroup internal_key
 * \brief Enum used in descriptors to specify the desired padding
 */
enum sdc_padding {
    SDC_PADDING_INTERNAL,   /**< Handled by kernel or arch specific part */
    SDC_PADDING_NO,         /**< No padding */
    SDC_PADDING_HIDDEN,     /**< No official padding, needed for workaround (DAL CTR operates on full blocks only) */
    SDC_PADDING_PKCS7      /**< Userspace to perform PKCS7 padding */
};

/**
 * \ingroup internal_key
 * \brief Type of enum used in descriptors to specify the desired padding
 */
typedef enum sdc_padding sdc_padding_t;

/**
 * \ingroup internal_key
 * \brief Interface of a function to add the desired amount of padding bytes to the buffer
 */
typedef sdc_error_t (*sdc_pad_add_func_t)(uint8_t *buffer, size_t offset, uint8_t padding_bytes);

/**
 * \ingroup internal_key
 * \brief Struct containing wrapped key data for import of keystore keys
 *
 * This struct will be used inside of architecture specific functions
 * to avoid passing a high number of arguments without changes through
 * multiple levels of hierarchy
 */
struct sdc_wrapped_key {
    sdc_wrap_unwrap_type_t *wrap_type; /**< wrap_type with which the key was wrapped */
    sdc_wrap_unwrap_desc_t *wrap_desc; /**< wrap_desc with which the key was wrapped */
    sdc_keysecret_enc_t enc; /**< key encoding before it was wrapped */
    uint8_t *iv; /**< IV with which the key was wrapped */
    size_t iv_len; /**< length of IV with which the key was wrapped */
    uint8_t *tag; /**< authentication data with which the key was wrapped */
    size_t tag_len; /**< length of authentication data with which the key was wrapped */
    uint8_t *wrapped_data; /**< wrapped key data */
    size_t wrapped_len; /**< length of wrapped key data */
};
/**
 * \ingroup internal_key
 * \brief Type for wrapped key data for import of keystore keys
 */
typedef struct sdc_wrapped_key sdc_wrapped_key_t;

/**
 * \ingroup internal_key
 * \brief Struct containing key modifier data for generating built-in keys
 *
 * This struct will be used inside of architecture specific functions
 * to avoid passing a high number of arguments without changes through
 * multiple levels of hierarchy
 */
struct sdc_key_modifier {
    uint8_t *data;  /**< modifier data */
    size_t len;     /**< modifier data length */
    bool is_secret; /**< true if data is secret and should be zeroed and freed as soon as possible */
};
/**
 * \ingroup internal_key
 * \brief Type for key modifier data for generating built-in keys
 */
typedef struct sdc_key_modifier sdc_key_modifier_t;

/**
 * \ingroup internal_key
 * \brief Struct containing key id range used for installation etc. of keystore keys
 *
 * This struct will be used inside of architecture specific functions
 * to avoid passing a high number of arguments without changes through
 * multiple levels of hierarchy
 */
struct sdc_key_id_range {
    sdc_key_id_t first;   /**< first (minimal) allowed KID */
    sdc_key_id_t last;    /**< last (maximal) allowed KID */
    sdc_key_id_t *result; /**< pointer to store result KID, NULL in case of explicit kid */
};
/**
 * \ingroup internal_key
 * \brief Type for key id range used for installation etc. of keystore keys
 */
typedef struct sdc_key_id_range sdc_key_id_range_t;

/**
 * \ingroup internal
 * \defgroup internal_descs Operation descriptors
 * Descriptors containing information about e.g. valid IV range of selected operation
 */

/**
 * \ingroup internal_descs
 * \brief Type to describe a value range - min/max/mod + default value
 */
struct sdc_range {
    size_t min;          /**< Minimal valid length */
    size_t max;          /**< Maximal valid length */
    size_t mod;          /**< If > 1 the result of length modulo this value has to be 0 */
    size_t dflt;         /**< Default length */
};
/**
 * \ingroup internal_descs
 * \brief Typedef of \ref sdc_range
 */
typedef struct sdc_range sdc_range_t;

/**
 * \ingroup internal_descs
 * \brief Type to describe max value and alignment required for plain or cipher inputs
 *
 * e.g. for RAW encrypt/decrypt plain and cipher need to be aligned - max according to arch limitations
 */
struct sdc_plain_cipher_spec {
    size_t min;   /**< length of data must be at least this value */
    size_t max;   /**< length of data must not exceed this value */
    bool aligned; /**< If true the overall processed data length needs to be a multiple of the block length */
};
/**
 * \ingroup internal_descs
 * \brief Typedef of \ref sdc_plain_cipher_spec
 */
typedef struct sdc_plain_cipher_spec sdc_plain_cipher_spec_t;

/**
 * \ingroup internal_descs
 * \brief Type to describe input output requirements of the operation (e.g. encrypt/decrypt/wrap/unwrap)
 *
 * e.g. What is the block alignment of the algorithm (16 for AES)
 * What is the max length of data the architecture can process at once (max_chunk_len) ?
 * What kind of padding is used ?
 */
struct sdc_inout_spec {
    size_t block_len;       /**< Block length used by the algorithm */
    size_t max_chunk_len;   /**< Max data length a single processing call (e.g. update ioctl) can process at once
                                 If \ref chunk_len_aligned == true this value needs to be a multiple of \ref block_len */
    bool chunk_len_aligned; /**< If true the size processed by a single processing call (e.g. update ioctl) needs
                                 to be a multiple of \ref block_len */

    sdc_plain_cipher_spec_t plain;  /**< max and alignment definition for plain data */
    sdc_plain_cipher_spec_t cipher; /**< max and alignment definition for cipher data */

    sdc_padding_t padding;  /**< select padding */
};
/**
 * \ingroup internal_descs
 * \brief Typedef of \ref sdc_inout_spec
 */
typedef struct sdc_inout_spec sdc_inout_spec_t;

/**
 * \ingroup internal_descs
 * \brief Type to describe input requirements of the operation (e.g. sign/verify/dgst)
 *
 * e.g. What is the block alignment of the algorithm (16 for AES)
 * What is the max length of data the architecture can process at once (max_chunk_len) ?
 * What kind of padding is used ?
 */
struct sdc_in_spec {
    size_t block_len;       /**< Block length used by the algorithm */
    size_t max_chunk_len;   /**< Max data length a single processing call (e.g. update ioctl) can process at once
                                 If \ref chunk_len_aligned == true this value needs to be a multiple of \ref block_len */
    bool chunk_len_aligned; /**< If true the size processed by a single processing call (e.g. update ioctl) needs
                                 to be a multiple of \ref block_len */

    sdc_plain_cipher_spec_t plain;  /**< max and alignment definition for plain data */

    sdc_padding_t padding;  /**< select padding */
};
/**
 * \ingroup internal_descs
 * \brief Typedef of \ref sdc_in_spec
 */
typedef struct sdc_in_spec sdc_in_spec_t;

/**
 * \ingroup internal_descs
 * \brief Description of relevant parameters for a \ref sdc_wrap_unwrap_type_t
 *
 * \todo document
 * \todo add missing fields
 */
struct sdc_wrap_unwrap_desc {
    sdc_inout_spec_t data; /**< block length, max sizes, padding specification for data */

    sdc_range_t iv; /**< values for IV */
    sdc_range_t tag; /**< values for tag */

    bool supports_iuf; /**< init update finalize API is supported - some algs require all data at once */
};
/**
 * \ingroup internal_descs
 * \brief Typedef of \ref sdc_wrap_unwrap_desc
 */
typedef struct sdc_wrap_unwrap_desc sdc_wrap_unwrap_desc_t;

/**
 * \ingroup internal_descs
 * \brief Description of relevant parameters for a \ref sdc_encrypt_decrypt_type_t
 *
 * \todo document
 * \todo add missing fields
 */
struct sdc_encrypt_decrypt_desc {
    sdc_inout_spec_t data; /**< block length, max sizes, padding specification for data */

    sdc_range_t iv; /**< values for IV */

    bool supports_iuf; /**< init update finalize API is supported - some algs require all data at once */
};
/**
 * \ingroup internal_descs
 * \brief Typedef of \ref sdc_encrypt_decrypt_desc
 */
typedef struct sdc_encrypt_decrypt_desc sdc_encrypt_decrypt_desc_t;

/**
 * \ingroup internal_descs
 * \brief Description of relevant parameters for a \ref sdc_sign_verify_type_t
 *
 * \todo document
 * \todo add missing fields
 */
struct sdc_sign_verify_desc {
    sdc_in_spec_t data; /**< block length, max sizes, padding specification for data */

    sdc_range_t iv; /**< values for IV */
    sdc_range_t tag; /**< values for tag */

    bool supports_iuf; /**< init update finalize API is supported - some algs require all data at once */
};
/**
 * \ingroup internal_descs
 * \brief Typedef of \ref sdc_sign_verify_desc
 */
typedef struct sdc_sign_verify_desc sdc_sign_verify_desc_t;

/**
 * \ingroup internal_descs
 * \brief Description of relevant parameters for a \ref sdc_dgst_type_t
 *
 * \todo document
 * \todo add missing fields
 */
struct sdc_dgst_desc {
    sdc_in_spec_t data; /**< block length, max sizes, padding specification for data */

    sdc_range_t digest; /**< values for digest */

    bool supports_iuf; /**< init update finalize API is supported - some algs require all data at once */
};


/**
 * \ingroup internal
 * \defgroup internal_key_info Key Info
 * Types, enums etc. relevant for the key info filled by \ref sdc_arch_session_key_info_get
 */

/**
 * \ingroup internal_key_info
 * \brief enum to describe the type of key
 */
typedef enum sdc_key_info_type {
    SDC_KEY_TYPE_INVALID, /**< invalid key (e.g. unset key) */
    SDC_KEY_TYPE_KEYSTORE_PLAIN, /**< a keystore key using a plain key secret */
    SDC_KEY_TYPE_KEYSTORE_RANDOM, /**< a keystore key using a random generated key secret */
    SDC_KEY_TYPE_KEYSTORE_PRODUCT, /**< a special kind of keystore key using a plain key secret - i.e. common key secret for all devices of a product */
    SDC_KEY_TYPE_KEYSTORE_IMPORTED, /**< a special kind of keystore key using a plain key secret - i.e. common key secret for all devices of a product */
    SDC_KEY_TYPE_BUILT_IN, /**< a key derived from the device specific internal secret - available even if keystore is not restored */
    SDC_KEY_TYPE_SESSION /**< a session specific key - e.g. imported to the current session */
} sdc_key_info_type_t;

/**
 * \ingroup internal_key_info
 * \brief enum to describe the authenticity of the key
 */
typedef enum sdc_key_authenticity {
    SDC_KEY_AUTH_UNKNOWN, /**< authenticity could not be determined */
    SDC_KEY_AUTH_LOW, /**< low authenticity */
    SDC_KEY_AUTH_MEDIUM, /**< medium authenticity */
    SDC_KEY_AUTH_HIGH /**< high authenticity - i.e. builtin key(\ref SDC_KEY_TYPE_BUILT_IN) or key being imported using a key with high authenticity */
} sdc_key_authenticity_t;

/**
 * \ingroup internal_key_info
 * \brief enum to describe the confidentiality of the key
 */
typedef enum sdc_key_confidentiality {
    SDC_KEY_CONFIDENT_UNKNOWN, /**< confidentiality could not be determined */
    SDC_KEY_CONFIDENT_LOW, /**< low confidentiality - e.g. keys generated using plain key secret(\ref SDC_KEY_TYPE_KEYSTORE_PLAIN)*/
    SDC_KEY_CONFIDENT_MEDIUM, /**< medium confidentiality - e.g. \ref SDC_KEY_TYPE_KEYSTORE_PRODUCT keys, \ref SDC_KEY_TYPE_KEYSTORE_RANDOM keys using user-space random, imported keys */
    SDC_KEY_CONFIDENT_HIGH /**< high confidentiality - i.e. builtin key(\ref SDC_KEY_TYPE_BUILT_IN) or \ref SDC_KEY_TYPE_KEYSTORE_RANDOM keys using kernel-space random */
} sdc_key_confidentiality_t;

/**
 * \ingroup internal_key_info
 * \brief struct filled by \ref sdc_arch_session_key_info_get
 *
 * As key info API is currently not external available this struct is only defined internally
 */
typedef struct sdc_key_info {
    sdc_key_info_type_t key_type; /**< type of the key */

    sdc_key_fmt_t key_fmt; /**< type of the key secret */

    size_t key_len_bits; /**< length of the key in bits */
    size_t key_len_bytes; /**< length of the key in bytes */

    sdc_key_id_t kid; /**< KID of a keystore key (\ref SDC_KEY_TYPE_KEYSTORE_PLAIN, \ref SDC_KEY_TYPE_KEYSTORE_RANDOM, \ref SDC_KEY_TYPE_KEYSTORE_PRODUCT). For other key types \ref SDC_FLAG_INVALID_KID.*/

    struct sdc_permissions perms; /**< struct storing the permission masks as well as the gid */

    sdc_key_modifier_t key_modifier; /**< struct storing key modifier data */

    sdc_key_authenticity_t authenticity; /**< authenticity classification of the key */
    sdc_key_confidentiality_t confidentiality; /**< confidentiality classification of the key */
} sdc_key_info_t;

/**
 * \ingroup internal
 * \defgroup internal_formatted Read/Write formatted data
 * defines, types and enums ... needed for formatted output
 * formatted data consists out of
 * formatted header followed by
 * additional_key_info,
 *
 * format data items
 * e.g. wrap
 * ----------------------------
 * data,
 * iv,
 * tag,
 * aad
 * ----------------------------
 */

/**
 * \ingroup internal_formatted
 * \defgroup internal_formatted_api Formatted API
 * SDC internal functions and typed to read/write formatted data
 */
/**
 * \ingroup internal_formatted_api
 * \defgroup internal_formatted_api_read Functions to read formatted data
 */
/**
 * \ingroup internal_formatted_api
 * \defgroup internal_formatted_api_write Functions to write formatted data
 */
/**
 * \ingroup internal_formatted_api
 * \defgroup internal_formatted_api_autoload Functions to automatically load the key specified in the formatted data
 */

/**
 * \ingroup internal_formatted
 * \defgroup internal_formatted_external Architecture independent format
 * Defines, types(packed) defining the structure/values of the architecture independent format
 */

/**
 * \ingroup internal_formatted_external
 * \brief magic used to mark formatted data
 */
#define SDC_FORMATTED_MAGIC 0xAD175DCF

/**
 * \ingroup internal_formatted_api
 * \brief key kind definitions for formatted output
 */
typedef enum sdc_form_key_type {
    SDC_FORMATTED_KEY_FIRST = 0,
    SDC_FORMATTED_KEY_STORAGE = SDC_FORMATTED_KEY_FIRST,
    SDC_FORMATTED_KEY_SESSION, /* currently not supported */
    SDC_FORMATTED_KEY_BUILT_IN, /* currently not supported */
    SDC_FORMATTED_KEY_LAST = SDC_FORMATTED_KEY_BUILT_IN
} sdc_form_key_type_t;

/**
 * \ingroup internal_formatted_external
 * \brief common part of externally used formatted key header - version 0
 */
typedef struct sdc_form_header_key_info_common_v0 {
    uint32_t key_type; /**< numeric encoding of \ref sdc_form_key_type_t */
} __attribute__((packed)) sdc_form_header_key_info_common_v0_t;

/**
 * \ingroup internal_formatted_external
 * \brief keystore specific part of externally used formatted key header - version 0
 */
typedef struct sdc_form_header_key_info_keystore_v0 {
    uint32_t kid; /**< key idenifier if \p key_type is SDC_FORMATTED_KEY_STORAGE */
} __attribute__((packed)) sdc_form_header_key_info_keystore_v0_t;

/**
 * \ingroup internal_formatted_external
 * \brief mark the uid as invalid - mapped to uid_t -1
 */
#define SDC_INTERN_FORMATTED_INVAL_UID 0xFFFFFFFF

/**
 * \ingroup internal_formatted_external
 * \brief mark the gid as invalid - mapped to gid_t -1
 */
#define SDC_INTERN_FORMATTED_INVAL_GID 0xFFFFFFFF

/**
 * \ingroup internal_formatted_external
 * \brief builtin specific part of externally used formatted key header - version 0
 */
typedef struct sdc_form_header_key_info_builtin_v0 {
    uint32_t secret_type; /**< type of the key secret */
    uint32_t key_len_bits; /**< length of key in bits */
    uint32_t key_len_bytes; /**< length of key in bytes */
    uint32_t uid; /**< uid allowed to perform operations specified in \ref perms_owner */
    uint32_t gid; /**< gid allowed to perform operations specified in \ref perms_group */
    uint32_t perms_owner; /**< mask defining the operations allowed for \ref uid */
    uint32_t perms_group; /**< mask defining the operations allowed for \ref gid */
    uint32_t perms_others; /**< mask defining the operations allowed for anybody who is neither the \ref uid nor a member of \ref gid*/
    uint32_t perms_inherit_owner; /**< upper limit for \ref perms_owner of the new key when importing a key using this key */
    uint32_t perms_inherit_group; /**< upper limit for \ref perms_group of the new key when importing a key using this key */
    uint32_t perms_inherit_others; /**< upper limit for \ref perms_others of the new key when importing a key using this key */
    uint32_t modifier_len; /**< length of the key modifier - 0 if no modifier or secret modifer (\ref secret_modifier) */
    uint8_t secret_modifier; /**< the key has a modifier, but this modifier contains secret data - therefore not included */
} __attribute__((packed)) sdc_form_header_key_info_builtin_v0_t;

/**
 * \ingroup internal_formatted_api
 * \brief common part of internally used formatted key header
 */
typedef struct sdc_form_header_key_info_common_generic {
    sdc_form_key_type_t type; /**< type of the key in the formatted output */
} sdc_form_header_key_info_common_generic_t;

/**
 * \ingroup internal_formatted_api
 * \brief keystore specific part of internally used formatted key header
 */
typedef struct sdc_form_header_key_info_keystore_generic {
    sdc_key_id_t kid; /**< key idenifier if \p key_type is SDC_FORMATTED_KEY_STORAGE */
} sdc_form_header_key_info_keystore_generic_t;

/**
 * \ingroup internal_formatted_api
 * \brief builtin specific part of internally used formatted key header
 */
typedef struct sdc_form_header_key_info_builtin_generic {
    sdc_key_fmt_t key_fmt; /**< type of the key secret */

    size_t key_len_bits; /**< length of key in bits */
    size_t key_len_bytes; /**< length of key in bytes */

    sdc_permissions_t perms; /**< struct storing the permission masks as well as the gid */

    sdc_key_modifier_t key_modifier; /**< struct storing key modifier data */
} sdc_form_header_key_info_builtin_generic_t;

/**
 * \ingroup internal_formatted_api
 * \brief internally used formatted key header
 */
typedef struct sdc_form_header_key_info_generic {
    sdc_form_header_key_info_common_generic_t common; /**< part of key header common to all keys */
    union {
        sdc_form_header_key_info_keystore_generic_t keystore; /**< part of key header only relevant for keystore keys */
        sdc_form_header_key_info_builtin_generic_t builtin; /**< part of builtin header only relevant for keystore keys */
    };
} sdc_form_header_key_info_generic_t;


/**
 * \ingroup internal_formatted_api
 * \brief enum to specify if the formatted data belongs to wrap, encrypt ...
 */
typedef enum sdc_form_operation {
    SDC_FORMATTED_TYPE_FIRST = 0,
    SDC_FORMATTED_TYPE_ENCRYPT_DECRYPT = SDC_FORMATTED_TYPE_FIRST,
    SDC_FORMATTED_TYPE_SIGN_VERIFY,
    SDC_FORMATTED_TYPE_WRAP_UNWRAP,
    SDC_FORMATTED_TYPE_LAST = SDC_FORMATTED_TYPE_WRAP_UNWRAP
} sdc_form_operation_t;

/**
 * \ingroup internal_formatted_external
 * \brief externally used formatted type header for wrap unwrap - version 0
 */
typedef struct sdc_form_header_wrap_unwrap_v0 {
    uint32_t alg; /**< numeric encoding of \ref sdc_wrap_unwrap_alg_t */
    uint32_t blk; /**< numeric encoding of \ref sdc_wrap_unwrap_blk_t */
    uint32_t data_len; /**< length of data in the formatted output */
    uint32_t iv_len; /**< length of iv in the formatted output */
    uint32_t tag_len; /**< length of tag in the formatted output */
    uint32_t aad_len; /**< length of aad data in the formatted output */
} __attribute__((packed)) sdc_form_header_wrap_unwrap_v0_t;

/**
 * \ingroup internal_formatted_external
 * \brief externally used formatted type header for encrypt decrypt - version 0
 */
typedef struct sdc_form_header_encrypt_decrypt_v0 {
    uint32_t alg; /**< numeric encoding of \ref sdc_encrypt_decrypt_alg_t */
    uint32_t blk; /**< numeric encoding of \ref sdc_encrypt_decrypt_blk_t */
    uint32_t data_len; /**< length of data in the formatted output */
    uint32_t iv_len; /**< length of iv in the formatted output */
} __attribute__((packed)) sdc_form_header_encrypt_decrypt_v0_t;

/**
 * \ingroup internal_formatted_external
 * \brief externally used formatted type header for sign verify - version 0
 */
typedef struct sdc_form_header_sign_verify_v0 {
    uint32_t alg; /**< numeric encoding of \ref sdc_sign_verify_alg_t */
    uint32_t hash; /**< numeric encoding of \ref sdc_sign_verify_hash_t */
    uint32_t data_len; /**< length of data in the formatted output */
    uint32_t iv_len; /**< length of iv in the formatted output */
    uint32_t tag_len; /**< length of tag in the formatted output */
} __attribute__((packed)) sdc_form_header_sign_verify_v0_t;

/**
 * \ingroup internal_formatted_api
 * \brief internally used formatted type header for wrap unwrap
 */
typedef struct sdc_form_header_wrap_unwrap_generic {
    sdc_wrap_unwrap_alg_t alg; /**< algorithm used for formatted output */
    sdc_wrap_unwrap_blk_t blk; /**< block mode used for formatted output */
    uint8_t *data; /**< pointer to the data section of the formatted output */
    size_t data_len; /**< length of the data section in the formatted output */
    uint8_t *iv; /**< pointer to the iv section of the formatted output */
    size_t iv_len; /**< length of the iv section in the formatted output */
    uint8_t *tag; /**< pointer to the tag section of the formatted output */
    size_t tag_len; /**< length of the tag section in the formatted output */
    uint8_t *aad; /**< pointer to the aad data section of the formatted output */
    size_t aad_len; /**< length of the aad data section in the formatted output */
} sdc_form_header_wrap_unwrap_generic_t;

/**
 * \ingroup internal_formatted_api
 * \brief internally used formatted type header for encrypt decrypt
 */
typedef struct sdc_form_header_encrypt_decrypt_generic {
    sdc_encrypt_decrypt_alg_t alg; /**< algorithm used for formatted output */
    sdc_encrypt_decrypt_blk_t blk; /**< block mode used for formatted output */
    uint8_t *data; /**< pointer to the data section of the formatted output */
    size_t data_len; /**< length of the data section in the formatted output */
    uint8_t *iv; /**< pointer to the iv section of the formatted output */
    size_t iv_len; /**< length of the iv section in the formatted output */
} sdc_form_header_encrypt_decrypt_generic_t;

/**
 * \ingroup internal_formatted_api
 * \brief internally used formatted type header for sign verify
 */
typedef struct sdc_form_header_sign_verify_generic {
    sdc_sign_verify_alg_t alg; /**< alg used for formatted output */
    sdc_sign_verify_hash_t hash; /**< hash mode used for formatted output */
    uint8_t *data; /**< pointer to the data section of the formatted output */
    size_t data_len; /**< length of the data section in the formatted output */
    uint8_t *iv; /**< pointer to the iv section of the formatted output */
    size_t iv_len; /**< length of the iv section in the formatted output */
    uint8_t *tag; /**< pointer to the tag section of the formatted output */
    size_t tag_len; /**< length of the tag section in the formatted output */
} sdc_form_header_sign_verify_generic_t;

/**
 * \ingroup internal_formatted_external
 * \brief externally used formatted header
 */
typedef struct sdc_form_header_ext {
    uint32_t magic; /**< \ref SDC_FORMATTED_MAGIC - used for the first check if the data is a formatted output */
    uint32_t key_info_version; /**< version information of the included key info */
    uint32_t key_info_len; /**< length of the included key info */
    uint32_t format_type_version; /**< version information of the included type info */
    uint32_t format_type; /**< numeric value of the type (\ref sdc_form_operation_t) */
    uint32_t format_len; /**< length of the included type info */
} __attribute__((packed)) sdc_form_header_ext_t;

/**
 * \ingroup internal_formatted_api
 * \brief internally used formatted header
 */
typedef struct sdc_form_header_generic {
    sdc_form_header_key_info_generic_t key_info; /**< information on the key */
    sdc_form_operation_t operation; /**< the operation encrypt/decrypt, wrap/unwrap or sign/verify of the formatted output */
    union {
        sdc_form_header_wrap_unwrap_generic_t wrap_unwrap; /**< wrap/unwrap specific information */
        sdc_form_header_encrypt_decrypt_generic_t encrypt_decrypt; /**< encrypt/decrypt specific information */
        sdc_form_header_sign_verify_generic_t sign_verify; /**< sign/verify specific information */
    };
    size_t header_len; /**< length of the header */
    size_t overall_formatted_len; /**< overall length of the formatted output */
} sdc_form_header_generic_t;


/* Functions */

/**
 * \ingroup internal_formatted_api
 * \brief Initialize header of structure
 *
 * \param[in] header
 */
void sdc_inter_form_header_init(sdc_form_header_generic_t *header);

/**
 * \ingroup internal_formatted_api
 * \brief Deinitialize header of structure
 *
 * This includes freeing the modifier of the key info for builtin keys
 *
 * \param[in] header
 */
void sdc_inter_form_header_free(sdc_form_header_generic_t *header);

/**
 * \ingroup internal_formatted_api_write
 * \brief Wrap/Unwrap: Fill the header with length and format information
 *
 * \param[in] session current session
 * \param[in] type \ref sdc_wrap_unwrap_type_t to write to header
 * \param[in] data_len
 * \param[in] iv_len
 * \param[in] tag_len
 * \param[in] aad_len
 * \param[out] header pointer to the header struct
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_INTERN_API sdc_error_t sdc_intern_wrap_unwrap_formatted_fill_header (sdc_session_t *session,
                                                                         const sdc_wrap_unwrap_type_t *type,
                                                                         size_t data_len,
                                                                         size_t iv_len,
                                                                         size_t tag_len,
                                                                         size_t aad_len,
                                                                         sdc_form_header_generic_t *header);

/**
 * \ingroup internal_formatted_api_write
 * \brief Wrap/Unwrap: Update wrap/unwrap data, iv, sig, and aad pointers in \p header
 *
 * This functions needs to be called before using pointers in \ref sdc_form_header_generic_t
 * to read or write the corresponding sections.
 * In case of generating a formatted output this function needs to be called after
 * setting the length values of the header using \ref sdc_intern_wrap_unwrap_formatted_fill_header.
 * In case of unwrapping a formatted output this function needs to be called after
 * setting the length values of the header using \ref sdc_intern_formatted_read_header
 *
 * \param[out] header pointer to the header struct
 * \param[in] buf_ptr pointer to buffer having exactly \ref sdc_form_header_generic_t -> \p overall_formatted_len bytes
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_INTERN_API sdc_error_t sdc_intern_wrap_unwrap_formatted_update_pointers (
    sdc_form_header_generic_t *header,
    uint8_t *buf_ptr);

/**
 * \ingroup internal_formatted_api_write
 * \brief Encrypt/Decrypt: Fill the header with length and format information
 *
 * \param[in] session current session
 * \param[in] type \ref sdc_encrypt_decrypt_type_t to write to header
 * \param[in] data_len
 * \param[in] iv_len
 * \param[out] header pointer to the header struct
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_INTERN_API sdc_error_t sdc_intern_encrypt_decrypt_formatted_fill_header (sdc_session_t *session,
                                                                             const sdc_encrypt_decrypt_type_t *type,
                                                                             size_t data_len,
                                                                             size_t iv_len,
                                                                             sdc_form_header_generic_t *header);

/**
 * \ingroup internal_formatted_api_write
 * \brief Encrypt/Decrypt: Update encrypt/decrypt data, and iv pointers in \p header
 *
 * This functions needs to be called before using pointers in \ref sdc_form_header_generic_t
 * to read or write the corresponding sections.
 * In case of generating a formatted output this function needs to be called after
 * setting the length values of the header using \ref sdc_intern_encrypt_decrypt_formatted_fill_header.
 * In case of decrypting a formatted output this function needs to be called after
 * setting the length values of the header using \ref sdc_intern_formatted_read_header
 *
 * \param[out] header pointer to the header struct
 * \param[in] buf_ptr pointer to buffer having exactly \ref sdc_form_header_generic_t -> \p overall_formatted_len bytes
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_INTERN_API sdc_error_t sdc_intern_encrypt_decrypt_formatted_update_pointers (
    sdc_form_header_generic_t *header,
    uint8_t *buf_ptr);


/**
 * \ingroup internal_formatted_api_write
 * \brief Sign/Verify: Fill the header with length and format information
 *
 * \param[in] session current session
 * \param[in] type \ref sdc_sign_verify_type_t to write to header
 * \param[in] data_len
 * \param[in] iv_len
 * \param[in] tag_len
 * \param[out] header pointer to the header struct
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_INTERN_API sdc_error_t sdc_intern_sign_verify_formatted_fill_header (sdc_session_t *session,
                                                                         const sdc_sign_verify_type_t *type,
                                                                         size_t data_len,
                                                                         size_t iv_len,
                                                                         size_t tag_len,
                                                                         sdc_form_header_generic_t *header);

/**
 * \ingroup internal_formatted_api_write
 * \brief Sign/Verify: Update sign/verify data, iv, and tag pointers in \p header
 *
 * This functions needs to be called before using pointers in \ref sdc_form_header_generic_t
 * to read or write the corresponding sections.
 * In case of generating a formatted output this function needs to be called after
 * setting the length values of the header using \ref sdc_intern_sign_verify_formatted_fill_header.
 * In case of verifying a formatted output this function needs to be called after
 * setting the length values of the header using \ref sdc_intern_formatted_read_header
 *
 * \param[out] header pointer to the header struct
 * \param[in] buf_ptr pointer to buffer having exactly \ref sdc_form_header_generic_t -> \p overall_formatted_len bytes
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_INTERN_API sdc_error_t sdc_intern_sign_verify_formatted_update_pointers (
    sdc_form_header_generic_t *header,
    uint8_t *buf_ptr);

/**
 * \ingroup internal_formatted_api_write
 * \brief Write the header of to the buffer of the formatted output
 *
 * When generating a formatted output this function will be used to write
 * the header information to the buffer for the formatted output.
 * This function won't write any data.
 * Before calling this function the header needs to be initialized using
 *  - encrypt/decrypt   : \ref sdc_intern_encrypt_decrypt_formatted_fill_header
 *  - wrap/unwrap       : \ref sdc_intern_wrap_unwrap_formatted_fill_header
 *  - sign/verify       : \ref sdc_intern_sign_verify_formatted_fill_header
 *
 * \param[in] header header to write to the buffer
 * \param[out] buf_ptr pointer to buffer having exactly \ref sdc_form_header_generic_t -> \p overall_formatted_len bytes
 *
 * \return \ref SDC_OK - no error
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_INTERN_API sdc_error_t sdc_intern_formatted_write_header (
    const sdc_form_header_generic_t *header,
    void *buf_ptr);

/**
 * \ingroup internal_formatted_api_read
 * \brief Read the header from the formatted output
 *
 * When reading a formatted output this function will be used to read
 * the header information from the formatted output.
 * This function won't read any data or update data pointers
 *
 * \param[out] gen_header header to fill according to the information of the formatted output
 * \param[in] formatted_ptr pointer to the formatted output
 * \param[in] formatted_len length of the formatted output
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_FORMATTED_DATA_INVALID - invalid formatted output
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_INTERN_API sdc_error_t sdc_intern_formatted_read_header (
    sdc_form_header_generic_t *gen_header,
    const void *formatted_ptr, size_t formatted_len);

/**
 * \ingroup internal
 * \brief Add pkcs7 padding to the buffer
 *
 * \param[out] buffer - buffer to be padded (length has to be >= offset + padding_bytes)
 * \param[in] offset - the offset to start padding == amount of valid data bytes in the buffer
 * \param[in] padding_bytes - amount of padding bytes to add
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PADDING_ERROR - padding_bytes is invalid for pkcs7
 */
SDC_INTERN_API sdc_error_t sdc_intern_add_pkcs7_padding(uint8_t *buffer,
                                                        size_t offset,
                                                        uint8_t padding_bytes);

/**
 * \ingroup internal
 * \brief Remove pkcs7 padding from the buffer
 *
 * \param[in,out] buffer - buffer containing data and padding
 * \param[in,out] data_len - input length of the buffer including padding
 *                   output length of data without padding
 * \param[in] block_alignment - block alignment to check max padding length
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PADDING_ERROR - padding is not pkcs7
 */
SDC_INTERN_API sdc_error_t sdc_intern_remove_pkcs7_padding(uint8_t *buffer,
                                                           size_t *data_len, size_t block_alignment);

/**
 * \ingroup internal
 * \brief Check if the parameters for ordinary input buffers (e.g. indata, aad, formatted) are valid
 *
 * This function can only be used in case no special value is used to encode a default
 *
 * data must not be NULL if len if != 0
 *
 * \param[in] data pointer to buffer
 * \param[in] len  length of buffer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - check violated
 */
SDC_INTERN_API sdc_error_t sdc_intern_check_data_input_buffer (const uint8_t *data, size_t len);

/**
 * \ingroup internal
 * \brief Check if the parameters for tag or iv buffers are valid
 *
 * data must not be NULL if len if != 0
 * furthermore the length must not equal the default
 *
 * \param[in] data pointer to buffer
 * \param[in] len  length of buffer
 * \param[in] default_val the special value to specify the default
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - check violated
 */
SDC_INTERN_API sdc_error_t sdc_intern_check_tag_iv_input_buffer (const uint8_t *data, size_t len,
                                                                 size_t default_val);

/**
 * \ingroup internal
 * \brief Check and initialize output buffer for data, formatted output
 *
 * data and len must not be NULL
 * *data must not be != NULL
 * Initialize *data to NULL and *len to 0 if available
 *
 * \param[in,out] data pointer to pointer to buffer
 * \param[in,out] len  pointer to length of buffer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - check violated
 */
SDC_INTERN_API sdc_error_t sdc_intern_check_init_data_output_buffer (uint8_t **data, size_t *len);

/**
 * \ingroup internal
 * \brief Check and initialize output buffers for tag/digest or iv
 *
 * data and len must not be NULL
 *
 * both require an external length which might be a default
 * default values are only allowed if the buffer is not provided external
 *
 * In case *data is != NULL the buffer is about to be provided externally
 *
 * External provided buffers are only allowed if (external_input != NULL)
 * In this case external_input is used to return if the buffer is externally
 * provided
 *
 * In case buffer is not externally provided initialize *data to NULL and *len to 0
 *
 * \param[in,out] data pointer to pointer to buffer
 * \param[in,out] len  pointer to length of buffer
 * \param[out] internal_len copy of the length value before resetting
 * \param[in] default_val the special value to specify the default
 * \param[in] external_input return if the buffer is external input
 *      if == NULL external inputs are prohibited
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - check violated
 */
SDC_INTERN_API sdc_error_t sdc_intern_check_init_dgst_tag_iv_output_buffer (uint8_t **data,
                                                                            size_t *len,
                                                                            size_t *internal_len,
                                                                            size_t default_val,
                                                                            bool *external_input);

/**
 * \ingroup internal
 * \brief Check if length fulfills data spec
 *
 * Checks :
 * in_len >= spec->min
 * in_len <= spec->max
 * if spec->aligned - in_len % block_len == 0
 *
 * \param[in] in_len             value to check
 * \param[in] spec               pointer to \ref sdc_inout_spec_t
 * \param[in] plain_input        input is plain data (use plain spec else use cipher spec)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_IN_DATA_INVALID - check violated
 */
SDC_INTERN_API sdc_error_t sdc_intern_inout_input_check_min_max_align(
    size_t in_len,
    const sdc_inout_spec_t *spec,
    bool plain_input);

/**
 * \ingroup internal
 * \brief Check if length fulfills data spec
 *
 * Checks :
 * in_len >= spec->min
 * in_len <= spec->max
 * if spec->aligned - in_len % block_len == 0
 *
 * \param[in] in_len             value to check
 * \param[in] spec               pointer to \ref sdc_in_spec_t
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_IN_DATA_INVALID - check violated
 */
SDC_INTERN_API sdc_error_t sdc_intern_input_only_check_min_max_align(
    size_t in_len,
    const sdc_in_spec_t *spec);

/**
 * \ingroup internal
 * \brief Check if processing additional data does not exceed max
 *
 * Checks :
 * prev_in_len + curr_in_len <= spec->max
 *
 * \param[in] prev_in_len        Amount of data processed previously
 * \param[in] curr_in_len        Data to be processed by this update
 * \param[in] spec               pointer to \ref sdc_inout_spec_t
 * \param[in] plain_input        input is plain data (use plain spec else use cipher spec)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_IN_DATA_INVALID - check violated
 */
SDC_INTERN_API sdc_error_t sdc_intern_inout_input_check_update_wont_exceed_max(
    size_t prev_in_len,
    size_t curr_in_len,
    const sdc_inout_spec_t *spec,
    bool plain_input);

/**
 * \ingroup internal
 * \brief Check if processing additional data does not exceed max
 *
 * Checks :
 * prev_in_len + curr_in_len <= spec->max
 *
 * \param[in] prev_in_len        Amount of data processed previously
 * \param[in] curr_in_len        Data to be processed by this update
 * \param[in] spec               pointer to \ref sdc_in_spec_t
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_IN_DATA_INVALID - check violated
 */
SDC_INTERN_API sdc_error_t sdc_intern_input_only_check_update_wont_exceed_max(
    size_t prev_in_len,
    size_t curr_in_len,
    const sdc_in_spec_t *spec);

/**
 * \ingroup internal
 * \brief Check if length fulfills requirements
 *
 * Checks :
 * len >= val->min
 * len <= val->max
 * if val->mod > 1 - len % val->mod == 0
 *
 * \param[in] len value to check
 * \param[in] val pointer to \ref sdc_range_t
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - check violated
 */
SDC_INTERN_API sdc_error_t sdc_intern_range_min_max_mod_check(
    size_t len,
    const sdc_range_t *val);

/**
 * \ingroup internal
 * \brief Get min, max, mod, default from \ref sdc_range_t
 *
 * \param[in] val pointer to \ref sdc_range_t
 * \param[out] min_val if != NULL used to return the min value
 * \param[out] max_val if != NULL used to return the max value
 * \param[out] mod if != NULL used to return the modulo of valid values between min and max
 *                 0 or 1 in case every value is valid
 * \param[out] default_val if != NULL used to return the default value
 *
 * \return \ref SDC_OK - no error
 */
SDC_INTERN_API sdc_error_t sdc_intern_range_get_min_max_mod_dflt(
    const sdc_range_t *val,
    size_t *min_val,
    size_t *max_val,
    size_t *mod,
    size_t *default_val);

/**
 * \ingroup internal
 * \brief Check that no permission exceeds the max allowed range
 *
 * \return \ref SDC_PERM_INVALID - a permission exceeds the max allowed range
 */
SDC_INTERN_API sdc_error_t sdc_intern_check_permission_range(const sdc_permissions_t *perms);

/**
 * \ingroup internal
 * \brief Check that the key secret encoding is valid for the format
 *
 * \return \ref SDC_KEY_FMT_INVALID - invalid format
 * \return \ref SDC_KEY_ENC_INVALID - invalid encoding for the given format
 */
SDC_INTERN_API sdc_error_t sdc_intern_check_key_enc(sdc_keysecret_enc_t enc, sdc_key_fmt_t fmt);

/**
 * \ingroup internal
 * \brief Check that the format is a valid symmetric key format
 *
 * \return \ref SDC_KEY_FMT_INVALID - invalid or asymmetric format
 */
SDC_INTERN_API sdc_error_t sdc_intern_check_sym_key_fmt(sdc_key_fmt_t fmt);

/**
 * \ingroup internal_key_info
 * \brief overwrite key_info (just overwrite modifier even if != NULL)
 */
SDC_INTERN_API void sdc_intern_session_key_info_init_overwrite(sdc_key_info_t *key_info);

/**
 * \ingroup internal_key_info
 * \brief initialize key_info struct (free previous modifier) and call arch function
 */
SDC_INTERN_API sdc_error_t sdc_intern_session_key_info_get(sdc_session_t *session,
                                                           sdc_key_info_t *key_info, bool with_modifier);

/**
 * \ingroup internal_formatted_api_autoload
 * \brief Try to load the key described in the formatted memory buffer
 *
 * Load the key described in the formatted memory buffer.\n
 * Other kinds of keys will result in error!
 *
 * \p secret_mod_data must only provided if a key with secret modifier has been used when generating
 * the formatted data.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[in] secret_mod_data  pointer to the secret modifier (NULL if no modifier)
 * \param[in] secret_mod_len   length of the \p secret_mod_data
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NOT_SUPPORTED - Key kind is not supported
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_NOT_READY - key not ready - try again later
 * \return \ref SDC_KEY_LOCKED - key store currently locked - try again later
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_AUTOLOAD_KEY_UNSUPPORTED - the key type of the formatted data is not supported
 * \return \ref SDC_AUTOLOAD_KEY_WITH_SECRET - the key of the formatted data requires a secret modifier to be specified
 * \return \ref SDC_MODIFIER_INVALID - the secret modifier is invalid for the key or its length is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_INTERN_API sdc_error_t sdc_intern_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len);

/**
 *  Overwrite the buffer containing secret data with 0
 *
 *  The asm volatile with memory in the clobber list is intended
 *  to ensure that the compiler does not reorder instructions
 *  and removes memset in optimization.
 *
 *  This function will be used by \ref sdc_overwrite_secret
 *  but does not check validity of parameters.
 *
 * \param[in] secret    Buffer containing potentially secret data
 * \param[in] secretlen Length of the buffer
 */
SDC_INTERN_API void sdc_intern_overwrite_secret(uint8_t *secret, size_t secretlen);

/**
 * \ingroup helper
 * \brief set variables using descriptor
 *
 * Note: in case returned fmt is a public key format the corresponding
 *       private key format can be used as well
 *
 * \param[in]  key_desc                 descriptor
 * \param[out] sup_key_fmt_protect      if != NULL fill set supported key format - unprotect operation (sign, encrypt, wrap)
 * \param[out] sup_key_fmt_unprotect    if != NULL fill set supported key format - unprotect operation (verify, decrypt, unwrap)
 * \param[out] sup_key_lens             if != NULL fill bmsk with all supported key length
 * \param[out] dflt_key_len             if != NULL fill set default key length
 */
SDC_INTERN_API sdc_error_t sdc_intern_key_lens_fmt(
    const sdc_key_desc_t *key_desc,
    sdc_key_fmt_t *sup_key_fmt_protect,
    sdc_key_fmt_t *sup_key_fmt_unprotect,
    sdc_key_len_bmsk_t *sup_key_lens,
    sdc_key_len_t *dflt_key_len);

/**
 *  Return function callback to add the requested padding (\ref sdc_padding_t) to data
 *
 * \param[in] padding Desired padding
 * \param[out] pad_func Padding function callback
 */
SDC_INTERN_API sdc_error_t sdc_intern_add_padding_func(
    const sdc_padding_t padding,
    sdc_pad_add_func_t *pad_func);

/**
 *  Function which can be used by the architecture specific implementation
 *  to add padding to the buffer
 *
 * \param[in] padding       padding format
 * \param[in,out] buffer    buffer to be padded
 * \param[in] data_len      amount of valid data bytes in the buffer == the offset to start padding
 * \param[in] buffer_len    length of \p buffer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_PADDING_ERROR - buffer_len == data_len for pkcs7
 * \return \ref SDC_INTERNAL_ERROR if this function is called for \ref SDC_PADDING_INTERNAL or unknown \ref sdc_padding_t
 */
sdc_error_t sdc_intern_pad(
    const sdc_padding_t padding,
    uint8_t *buffer, size_t data_len, size_t buffer_len);

/**
 *  Function which can be used by the architecture specific implementation
 *  to remove the padding from the buffer
 *
 * \param[in] padding           padding format
 * \param[in] buffer            buffer
 * \param[in,out] data_len       input length of buffer - output length after padding is removed
 * \param[in] block_alignment   block alignment of the algorithm
 *
 * \return \ref SDC_OK in case padding could be removed without error
 * \return \ref SDC_PADDING_ERROR in case removal of padding failed due to error in padding (e.g. unexpected byte)
 * \return \ref SDC_INTERNAL_ERROR if this function is called for \ref SDC_PADDING_INTERNAL or unknown \ref sdc_padding_t
 */
SDC_INTERN_API sdc_error_t sdc_intern_unpad(
    const sdc_padding_t padding,
    uint8_t *buffer, size_t *data_len, size_t block_alignment);

/**
 *  Helper to calculate exact length for cipher output
 *
 *  For encrypt and wrap the common code needs to allocate memory to store the result.
 *  This data needs to be large enough to contain encrypted/wrapped plain data + padding
 *
 * \param[in] padding padding format
 * \param[in] plain_len length of the plain input for encrypt or wrap operation
 * \param[in] block_len block length of the algorithm
 * \param[out] cipher_len length of cipher output (encrypted/wrapped plain data + padding)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_INTERN_API sdc_error_t sdc_intern_get_cipher_len(
    const sdc_padding_t padding,
    size_t plain_len,
    size_t block_len,
    size_t *cipher_len);

/**
 *  Helper to calculate upper limit for plain output
 *
 *  For decrypt and unwrap the common code needs to allocate memory to store the result.
 *  When using userspace padding and \ref sdc_intern_unpad this data needs to be large
 *  enough to hold data + padding.
 *
 * \param[in] padding padding format
 * \param[in] cipher_len length of the cipher input for decrypt or unwrap operation
 * \param[in] block_len block length of the algorithm
 * \param[out] max_plain_len upper limit of plain output
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_INTERN_API sdc_error_t sdc_intern_get_max_plain_len(
    const sdc_padding_t padding,
    size_t cipher_len,
    size_t block_len,
    size_t *max_plain_len);

/**
 *  Helper to set alignment and max length of plain/cipher data according to the padding
 *
 * \param[in] padding Desired padding
 * \param[in] arch_min  min length of data supported by the architecture
 * \param[in] arch_max  max overall length of data supported by the architecture
 * \param[in] block_len block length of the algorithm
 * \param[out] plain if != NULL max and alignment will be set according to padding
 * \param[out] cipher if != NULL max and alignment will be set according to padding
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_INTERN_API sdc_error_t sdc_intern_get_plain_cipher_spec(
    const sdc_padding_t padding,
    size_t arch_min,
    size_t arch_max,
    size_t block_len,
    sdc_plain_cipher_spec_t *plain,
    sdc_plain_cipher_spec_t *cipher);

/**
 *  Helper to calculate upper limit for update function output
 *
 * \param[in] padding padding format
 * \param[in] plain_len length of the cipher/plain input for encrypt/decrypt operation
 * \param[in] block_len block length of the algorithm
 * \param[out] cipher_len upper limit of plain/chiper output
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INTERNAL_ERROR - this probably means that a new value has been added to sdc_padding_t without handling it here
 */
SDC_INTERN_API sdc_error_t sdc_intern_get_update_len(
    const sdc_padding_t padding,
    size_t plain_len,
    size_t block_len,
    size_t *cipher_len);

/**
 *  Helper to overwrite potential confidential session data
 *
 *  This function is called on error or
 *  when operation is finalized (data no longer needed)
 *
 * \param[in] session pointer to the session
 */
SDC_INTERN_API void sdc_intern_clear_session_confidential(
        sdc_session_t *session);

/**
 *  Helper to verify sequence and operation (INIT/UPDATE/FINALIZE or ONESHOT)
 *
 *  This function is called very beginning of any operation
 *
 * \param[in] session pointer to the session
 * \param[in] crypto_op type of crypto operation
 * \param[in] op_state type of operation(INIT/UPDATE/FINALIZE or ONESHOT)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_OP_SEQUENCE_INVALID - this means that the operation sequence is invalid
 */
SDC_INTERN_API sdc_error_t sdc_intern_ops_sequence_ctl_check(
        const sdc_session_t *session,
        sdc_op_crypto_t crypto_op,
        sdc_op_state_t op_state);

/**
 *  Helper to set sequence and operation(INIT/UPDATE/FINALIZE or ONESHOT)
 *
 *  This function is called end of any operation.
 *
 * \param[out] session pointer to the session
 * \param[in] crypto_op type of crypto operation
 * \param[in] op_state type of operation(INIT/UPDATE/FINALIZE or ONESHOT)
 *
 * \return \ref SDC_OK - no error
 */
SDC_INTERN_API sdc_error_t sdc_intern_ops_sequence_ctl_set(
        sdc_session_t *session,
        sdc_op_crypto_t crypto_op,
        sdc_op_state_t op_state);

#endif
